<?php
/**
 * Plugin Name: CrashCount Crash Finder Shortcode
 * Plugin URI: https://crashcountnyc.com/about/embeds/
 * Description: Adds CrashCount embed shortcodes + Gutenberg wrappers for search, council map, strip, and saved results.
 * Version: 0.5.0
 * Requires at least: 6.4
 * Requires PHP: 8.0
 * Author: CrashCount NYC
 * License: GPL-2.0-or-later
 * Text Domain: crashcount-crashfinder-shortcode
 */

if (!defined('ABSPATH')) {
    exit;
}

final class CrashCount_CrashFinder_Shortcode {
    private const VERSION = '0.5.0';
    private const PRIVACY_POLICY_URL = 'https://crashcountnyc.com/privacy/';
    private const TERMS_URL = 'https://crashcountnyc.com/terms/';
    private const SHORTCODE = 'crashcount_crashfinder';
    private const COUNCIL_MAP_SHORTCODE = 'crashcount_council_map_tout';
    private const STRIP_SHORTCODE = 'crashcount_crashcounter_strip';
    private const FIND_RESULTS_SHORTCODE = 'crashcount_find_results';
    private const COUNCIL_MAP_SUMMARY_PATH = '/data/embeds/council-map-tout.json';
    private const COUNCIL_MAP_SUMMARY_CACHE_SECONDS = 900;
    private const STRIP_SUMMARY_PATH = '/data/embeds/strip';
    private const STRIP_SUMMARY_CACHE_SECONDS = 900;
    private const INDEX_HANDLE = 'crashcount-crashfinder-search-index';
    private const CORE_HANDLE = 'crashcount-crashfinder-search-core';
    private const SCRIPT_HANDLE = 'crashcount-crashfinder-widget';
    private const STYLE_HANDLE = 'crashcount-crashfinder-widget';
    private const MAP_TOUT_STYLE_HANDLE = 'crashcount-council-map-tout-inline';
    private const STRIP_STYLE_HANDLE = 'crashcount-crashcounter-strip-inline';
    private const FIND_RESULTS_STYLE_HANDLE = 'crashcount-find-results-inline';
    private const BLOCK_EDITOR_HANDLE = 'crashcount-embed-block-editor';
    private const DEFAULT_TRACKING = [
        'source' => 'partner_embed',
        'utm_source' => 'partner',
        'utm_medium' => 'embed',
        'utm_campaign' => 'crashfinder-partnership',
        'utm_content' => '',
        'utm_term' => '',
    ];
    private static int $instance_count = 0;
    private static bool $map_tout_style_enqueued = false;
    private static bool $strip_style_enqueued = false;
    private static bool $find_results_style_enqueued = false;
    private static bool $block_editor_script_inlined = false;

    public function __construct() {
        add_shortcode(self::SHORTCODE, [$this, 'render_shortcode']);
        add_shortcode(self::COUNCIL_MAP_SHORTCODE, [$this, 'render_council_map_shortcode']);
        add_shortcode(self::STRIP_SHORTCODE, [$this, 'render_strip_shortcode']);
        add_shortcode(self::FIND_RESULTS_SHORTCODE, [$this, 'render_find_results_shortcode']);
        add_action('init', [$this, 'register_blocks']);
        add_action('admin_init', [$this, 'register_privacy_policy_content']);
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_block_editor_assets']);
    }

    public function render_shortcode(array $atts = []): string {
        $attrs = shortcode_atts(
            [
                'base_url' => 'https://crashcountnyc.com',
                'api_base' => 'https://mk1mrx11pg.execute-api.us-east-1.amazonaws.com',
                'version' => 'latest',
                'asset_base' => '',
                'script_src' => '',
                'style_src' => '',
                'index_src' => '',
                'asset_version' => self::VERSION,
                'eyebrow' => 'Crash Finder',
                'title' => 'Try Crash Finder',
                'subtitle' => 'Look up any street, school, address, or intersection to see how safe the streets are.',
                'placeholder' => 'Search NYC location',
                'class' => '',
            ],
            $atts,
            self::SHORTCODE
        );

        $this->enqueue_assets($attrs);

        self::$instance_count++;
        $id = 'crashcount-crashfinder-' . self::$instance_count;
        $tracking = $this->resolve_tracking_params();
        $class_tokens = preg_split('/\s+/', trim((string) $attrs['class'])) ?: [];
        $class_tokens = array_filter(array_map('sanitize_html_class', $class_tokens));
        $class_name = trim(implode(' ', array_merge(['crashcount-crashfinder-widget'], $class_tokens)));

        $base_url = untrailingslashit(esc_url_raw((string) $attrs['base_url']));
        if ($base_url === '') {
            $base_url = 'https://crashcountnyc.com';
        }
        $asset_base = untrailingslashit(esc_url_raw((string) ($attrs['asset_base'] ?? '')));
        if ($asset_base === '') {
            $asset_base = $base_url;
        }
        $index_src = esc_url_raw((string) ($attrs['index_src'] ?? ''));
        if ($index_src === '') {
            $index_src = $asset_base . '/js/crashfinder-search-index.js';
        }

        $data = [
            'data-crashcount-widget' => 'crashfinder',
            'data-base-url' => esc_url_raw((string) $attrs['base_url']),
            'data-api-base' => esc_url_raw((string) $attrs['api_base']),
            'data-version' => sanitize_text_field((string) $attrs['version']),
            'data-index-src' => esc_url_raw($index_src),
            'data-source' => $tracking['source'],
            'data-utm-source' => $tracking['utm_source'],
            'data-utm-medium' => $tracking['utm_medium'],
            'data-utm-campaign' => $tracking['utm_campaign'],
            'data-utm-content' => $tracking['utm_content'],
            'data-utm-term' => $tracking['utm_term'],
            'data-eyebrow' => sanitize_text_field((string) $attrs['eyebrow']),
            'data-title' => sanitize_text_field((string) $attrs['title']),
            'data-subtitle' => sanitize_text_field((string) $attrs['subtitle']),
            'data-placeholder' => sanitize_text_field((string) $attrs['placeholder']),
        ];

        $html_attrs = [];
        foreach ($data as $key => $value) {
            if ($value === '') {
                continue;
            }
            $html_attrs[] = sprintf('%s="%s"', esc_attr($key), esc_attr($value));
        }

        $forward_params = $tracking;
        $hidden_inputs = '<input type="hidden" name="auto" value="1" />';
        foreach ($forward_params as $name => $value) {
            if ($value === '') {
                continue;
            }
            $hidden_inputs .= sprintf(
                '<input type="hidden" name="%s" value="%s" />',
                esc_attr($name),
                esc_attr($value)
            );
        }

        $search_action = $base_url . '/find/';
        $fallback_markup = sprintf(
            '<div class="crashcount-widget__fallback" style="border:1px solid #d1d5db;border-radius:12px;padding:12px;max-width:760px;">'
            . '<p style="margin:0 0 6px;font-weight:700;">%1$s</p>'
            . '<p style="margin:0 0 10px;color:#475569;font-size:0.92rem;">%2$s</p>'
            . '<form action="%3$s" method="get" target="_top" style="display:flex;gap:8px;flex-wrap:wrap;">'
            . '<input type="search" name="q" placeholder="%4$s" aria-label="%1$s search" style="flex:1 1 260px;padding:9px 10px;border:1px solid #cbd5e1;border-radius:8px;" />'
            . '%5$s'
            . '<button type="submit" style="padding:9px 12px;border:1px solid #1d4ed8;background:#1d4ed8;color:#fff;border-radius:8px;font-weight:600;cursor:pointer;">Search</button>'
            . '</form>'
            . '</div>',
            esc_html((string) $attrs['title']),
            esc_html((string) $attrs['subtitle']),
            esc_url($search_action),
            esc_attr((string) $attrs['placeholder']),
            $hidden_inputs
        );

        return sprintf(
            '<div id="%s" class="%s" %s>%s</div>',
            esc_attr($id),
            esc_attr($class_name),
            implode(' ', $html_attrs),
            $fallback_markup
        );
    }

    public function render_council_map_shortcode(array $atts = []): string {
        $attrs = shortcode_atts(
            [
                'base_url' => 'https://crashcountnyc.com',
                'map_path' => '/council-district/?map-first=true',
                'summary_path' => self::COUNCIL_MAP_SUMMARY_PATH,
                'href' => '',
                'image_src' => '',
                'embed_src' => '',
                'title' => 'How safe are your streets?',
                'button_text' => 'Expand map',
                'headline' => '',
                'as_of_text' => '',
                'window' => '',
                'show_text' => 'true',
                'mode' => 'image',
                'width' => '400px',
                'height' => '400px',
                'aspect_ratio' => '1 / 1',
                'class' => '',
            ],
            $atts,
            self::COUNCIL_MAP_SHORTCODE
        );

        $this->enqueue_council_map_styles();

        self::$instance_count++;
        $id = 'crashcount-council-map-' . self::$instance_count;

        $class_tokens = preg_split('/\s+/', trim((string) $attrs['class'])) ?: [];
        $class_tokens = array_filter(array_map('sanitize_html_class', $class_tokens));
        $class_name = trim(implode(' ', array_merge(['crashcount-council-map-tout'], $class_tokens)));

        $base_url = untrailingslashit(esc_url_raw((string) $attrs['base_url']));
        if ($base_url === '') {
            $base_url = 'https://crashcountnyc.com';
        }
        $tracking = $this->resolve_tracking_params();
        $mode = strtolower(sanitize_key((string) $attrs['mode']));
        if (!in_array($mode, ['image', 'iframe'], true)) {
            $mode = 'image';
        }

        $width = $this->sanitize_css_size((string) $attrs['width'], '400px');
        $height = $this->sanitize_css_size((string) $attrs['height'], '400px');
        $aspect_ratio = $this->sanitize_css_aspect_ratio((string) $attrs['aspect_ratio'], '1 / 1');
        $wrapper_style = sprintf(
            '--crashcount-council-map-width:%s;--crashcount-council-map-height:%s;--crashcount-council-map-aspect-ratio:%s;',
            $width,
            $height,
            $aspect_ratio
        );

        $summary_path = trim((string) $attrs['summary_path']);
        if ($summary_path === '') {
            $summary_path = self::COUNCIL_MAP_SUMMARY_PATH;
        }
        if ($summary_path[0] !== '/') {
            $summary_path = '/' . $summary_path;
        }

        $window_key = sanitize_text_field((string) $attrs['window']);

        $headline_raw = trim((string) $attrs['headline']);

        $as_of_raw = trim((string) $attrs['as_of_text']);

        $summary = [];
        $needs_summary = ($window_key === '' || $headline_raw === '' || $as_of_raw === '');
        if ($needs_summary && !$this->is_local_host_url($base_url)) {
            $summary = $this->fetch_council_map_summary($base_url, $summary_path);
        }

        if ($window_key === '' && isset($summary['window_key'])) {
            $window_key = sanitize_text_field((string) $summary['window_key']);
        }
        if ($headline_raw === '' && isset($summary['headline'])) {
            $headline_raw = (string) $summary['headline'];
        }
        if ($headline_raw === '') {
            $headline_raw = 'NYC traffic safety map.';
        }
        if ($as_of_raw === '' && isset($summary['as_of_text'])) {
            $as_of_raw = (string) $summary['as_of_text'];
        }

        $headline_html = wp_kses_post($headline_raw);
        $as_of_html = wp_kses_post($as_of_raw);

        $show_text = $this->sanitize_bool((string) $attrs['show_text'], true);

        if ($mode === 'iframe') {
            $embed_src = esc_url_raw((string) $attrs['embed_src']);
            if ($embed_src === '') {
                $embed_src = $base_url . '/embed/council-map/';
            }

            $target_href = esc_url_raw((string) $attrs['href']);
            if ($target_href === '') {
                $map_path = trim((string) $attrs['map_path']);
                if ($map_path === '') {
                    $map_path = '/council-district/?map-first=true';
                }
                if ($map_path[0] !== '/') {
                    $map_path = '/' . $map_path;
                }
                $target_href = $base_url . $map_path;
            }
            if ($window_key !== '') {
                $target_href = add_query_arg('window', $window_key, $target_href);
            }
            if ($target_href !== '') {
                $embed_src = add_query_arg('map_url', $target_href, $embed_src);
            }

            foreach ($tracking as $key => $value) {
                if ($value === '') {
                    continue;
                }
                $embed_src = add_query_arg($key, $value, $embed_src);
            }

            return sprintf(
                '<section id="%1$s" class="%2$s crashcount-council-map-tout--frame" aria-label="%3$s" style="%4$s">'
                . '<iframe class="crashcount-council-map-tout__frame" src="%5$s" title="%6$s" loading="lazy" referrerpolicy="strict-origin-when-cross-origin"></iframe>'
                . '</section>',
                esc_attr($id),
                esc_attr($class_name),
                esc_attr((string) $attrs['title']),
                esc_attr($wrapper_style),
                esc_url($embed_src),
                esc_attr((string) $attrs['title'])
            );
        }

        $href = esc_url_raw((string) $attrs['href']);
        if ($href === '') {
            $map_path = trim((string) $attrs['map_path']);
            if ($map_path === '') {
                $map_path = '/council-district/?map-first=true';
            }
            if ($map_path[0] !== '/') {
                $map_path = '/' . $map_path;
            }
            $href = $base_url . $map_path;
        }
        if ($window_key !== '') {
            $href = add_query_arg('window', $window_key, $href);
        }
        foreach ($tracking as $key => $value) {
            if ($value === '') {
                continue;
            }
            $href = add_query_arg($key, $value, $href);
        }

        $image_src = esc_url_raw((string) $attrs['image_src']);
        if ($image_src === '') {
            $image_src = $base_url . '/images/embeds/council-map-preview.png';
        }

        $title = sanitize_text_field((string) $attrs['title']);
        $button_text = sanitize_text_field((string) $attrs['button_text']);
        if ($title === '') {
            $title = 'How safe are your streets?';
        }
        if ($button_text === '') {
            $button_text = 'Expand map';
        }

        $copy_markup = '';
        if ($show_text && ($headline_html !== '' || $as_of_html !== '')) {
            $copy_markup = '<div class="crashcount-council-map-tout__copy">';
            if ($headline_html !== '') {
                $copy_markup .= sprintf(
                    '<p class="crashcount-council-map-tout__headline">%s</p>',
                    $headline_html
                );
            }
            if ($as_of_html !== '') {
                $copy_markup .= sprintf(
                    '<p class="crashcount-council-map-tout__asof">%s</p>',
                    $as_of_html
                );
            }
            $copy_markup .= '</div>';
        }

        return sprintf(
            '<section id="%1$s" class="%2$s" aria-label="%3$s" style="%8$s">'
            . '<header class="crashcount-council-map-tout__header"><h3>%3$s</h3></header>'
            . '<a class="crashcount-council-map-tout__link" href="%4$s" target="_top" rel="noopener">'
            . '<img class="crashcount-council-map-tout__image" src="%5$s" alt="%6$s" loading="lazy" />'
            . '<span class="crashcount-council-map-tout__button">%7$s</span>'
            . '</a>'
            . '%9$s'
            . '</section>',
            esc_attr($id),
            esc_attr($class_name),
            esc_html($title),
            esc_url($href),
            esc_url($image_src),
            esc_attr($title . ' preview'),
            esc_html($button_text),
            esc_attr($wrapper_style),
            $copy_markup
        );
    }

    public function render_strip_shortcode(array $atts = []): string {
        $attrs = shortcode_atts(
            [
                'base_url' => 'https://crashcountnyc.com',
                'summary_path' => self::STRIP_SUMMARY_PATH,
                'layer' => 'citywide',
                'id' => 'nyc',
                'window' => 'auto',
                'show_as_of' => 'true',
                'class' => '',
            ],
            $atts,
            self::STRIP_SHORTCODE
        );

        $this->enqueue_strip_styles();

        self::$instance_count++;
        $id = 'crashcount-strip-' . self::$instance_count;

        $layer = $this->sanitize_strip_layer((string) $attrs['layer']);
        $geo_id = $this->sanitize_strip_geo_id((string) $attrs['id']);
        $requested_window = $this->sanitize_window_value((string) $attrs['window']);
        $show_as_of = $this->sanitize_bool((string) $attrs['show_as_of'], true);

        $class_tokens = preg_split('/\s+/', trim((string) $attrs['class'])) ?: [];
        $class_tokens = array_filter(array_map('sanitize_html_class', $class_tokens));
        $class_name = trim(implode(' ', array_merge(['crashcount-crashcounter-strip'], $class_tokens)));

        $base_url = untrailingslashit(esc_url_raw((string) $attrs['base_url']));
        if ($base_url === '') {
            $base_url = 'https://crashcountnyc.com';
        }

        $summary_path = trim((string) $attrs['summary_path']);
        if ($summary_path === '') {
            $summary_path = self::STRIP_SUMMARY_PATH;
        }
        if ($summary_path[0] !== '/') {
            $summary_path = '/' . $summary_path;
        }

        $summary = [];
        if (!$this->is_local_host_url($base_url)) {
            $summary = $this->fetch_strip_summary($base_url, $summary_path, $layer, $geo_id);
        }

        $window_data = [];
        $window_key = '';
        $window_label = '';
        if (is_array($summary) && !empty($summary)) {
            $windows = isset($summary['windows']) && is_array($summary['windows']) ? $summary['windows'] : [];
            if ($requested_window !== '' && $requested_window !== 'auto' && isset($windows[$requested_window])) {
                $window_key = $requested_window;
                $window_data = is_array($windows[$requested_window]) ? $windows[$requested_window] : [];
            } else {
                $window_key = sanitize_text_field((string) ($summary['window_key'] ?? ''));
                if ($window_key !== '' && isset($windows[$window_key])) {
                    $window_data = is_array($windows[$window_key]) ? $windows[$window_key] : [];
                }
            }

            if (empty($window_data) && !empty($windows)) {
                $first_window = reset($windows);
                $window_data = is_array($first_window) ? $first_window : [];
                $window_key = sanitize_text_field((string) ($window_data['window_key'] ?? $window_key));
            }
            if ($window_key === '') {
                $window_key = sanitize_text_field((string) ($summary['window_key'] ?? ''));
            }
            $window_label = sanitize_text_field((string) ($window_data['window_label'] ?? $summary['window_label'] ?? $window_key));
        }

        $metrics_source = [];
        if (isset($window_data['metrics']) && is_array($window_data['metrics'])) {
            $metrics_source = $window_data['metrics'];
        } elseif (isset($summary['metrics']) && is_array($summary['metrics'])) {
            $metrics_source = $summary['metrics'];
        }

        $crashes = (int) ($metrics_source['crashes'] ?? 0);
        $injuries = (int) ($metrics_source['injuries'] ?? 0);
        $serious = (int) ($metrics_source['serious_injuries'] ?? 0);
        $deaths = (int) ($metrics_source['deaths'] ?? 0);

        $geo_name = sanitize_text_field((string) ($summary['name'] ?? $this->fallback_strip_name($layer, $geo_id)));
        $headline = $geo_name;
        if ($window_label !== '') {
            $headline .= ' · ' . $window_label;
        }
        $headline .= ' · Crashes ' . number_format_i18n($crashes);
        $headline .= ' · Injuries ' . number_format_i18n($injuries);
        $headline .= ' · Serious ' . number_format_i18n($serious);
        $headline .= ' · Deaths ' . number_format_i18n($deaths);

        $target_path = sanitize_text_field((string) ($summary['target_url'] ?? '/'));
        if ($target_path === '') {
            $target_path = '/';
        }
        if ($target_path[0] !== '/') {
            $target_path = '/' . ltrim($target_path, '/');
        }
        $href = $base_url . $target_path;
        if ($window_key !== '') {
            $href = add_query_arg('window', $window_key, $href);
        }
        $tracking = $this->resolve_tracking_params();
        foreach ($tracking as $key => $value) {
            if ($value === '') {
                continue;
            }
            $href = add_query_arg($key, $value, $href);
        }

        $as_of_text = sanitize_text_field((string) ($summary['as_of_text'] ?? ''));
        $meta_html = '';
        if ($show_as_of && $as_of_text !== '') {
            $meta_html = sprintf('<p class="crashcount-strip__meta">%s</p>', esc_html($as_of_text));
        }

        return sprintf(
            '<section id="%1$s" class="%2$s">'
            . '<a class="crashcount-strip__link" href="%3$s" target="_top" rel="noopener">'
            . '<p class="crashcount-strip__headline">%4$s</p>'
            . '%5$s'
            . '</a>'
            . '</section>',
            esc_attr($id),
            esc_attr($class_name),
            esc_url($href),
            esc_html($headline),
            $meta_html
        );
    }

    public function render_find_results_shortcode(array $atts = []): string {
        $attrs = shortcode_atts(
            [
                'base_url' => 'https://crashcountnyc.com',
                'embed_path' => '/embed/find-results/',
                'url' => '',
                'sid' => '',
                'height' => '230px',
                'class' => '',
                'title' => 'Crash Finder Results',
            ],
            $atts,
            self::FIND_RESULTS_SHORTCODE
        );

        $this->enqueue_find_results_styles();

        self::$instance_count++;
        $id = 'crashcount-find-results-' . self::$instance_count;

        $base_url = untrailingslashit(esc_url_raw((string) $attrs['base_url']));
        if ($base_url === '') {
            $base_url = 'https://crashcountnyc.com';
        }

        $embed_path = trim((string) $attrs['embed_path']);
        if ($embed_path === '') {
            $embed_path = '/embed/find-results/';
        }
        if ($embed_path[0] !== '/') {
            $embed_path = '/' . $embed_path;
        }
        $src = $base_url . $embed_path;

        $sid = sanitize_text_field((string) $attrs['sid']);
        $url = esc_url_raw((string) $attrs['url']);
        if ($sid === '' && $url !== '') {
            $sid = sanitize_text_field((string) $this->extract_sid_from_url($url));
        }
        if ($sid !== '') {
            $src = add_query_arg('sid', $sid, $src);
        }
        if ($url !== '') {
            $src = add_query_arg('url', $url, $src);
        }

        $tracking = $this->resolve_tracking_params();
        foreach ($tracking as $key => $value) {
            if ($value === '') {
                continue;
            }
            $src = add_query_arg($key, $value, $src);
        }

        $class_tokens = preg_split('/\s+/', trim((string) $attrs['class'])) ?: [];
        $class_tokens = array_filter(array_map('sanitize_html_class', $class_tokens));
        $class_name = trim(implode(' ', array_merge(['crashcount-find-results'], $class_tokens)));
        $height = $this->sanitize_css_size((string) $attrs['height'], '230px');
        $title = sanitize_text_field((string) $attrs['title']);
        if ($title === '') {
            $title = 'Crash Finder Results';
        }

        return sprintf(
            '<section id="%1$s" class="%2$s">'
            . '<iframe class="crashcount-find-results__frame" src="%3$s" title="%4$s" loading="lazy" referrerpolicy="strict-origin-when-cross-origin" style="height:%5$s;"></iframe>'
            . '</section>',
            esc_attr($id),
            esc_attr($class_name),
            esc_url($src),
            esc_attr($title),
            esc_attr($height)
        );
    }

    public function register_blocks(): void {
        if (!function_exists('register_block_type')) {
            return;
        }
        $this->register_block_editor_script();

        register_block_type(
            'crashcount/crashfinder',
            [
                'api_version' => 2,
                'editor_script' => self::BLOCK_EDITOR_HANDLE,
                'render_callback' => function (array $attributes = []): string {
                    return $this->render_shortcode($attributes);
                },
                'attributes' => [
                    'base_url' => ['type' => 'string'],
                    'api_base' => ['type' => 'string'],
                    'version' => ['type' => 'string'],
                    'asset_base' => ['type' => 'string'],
                    'eyebrow' => ['type' => 'string'],
                    'title' => ['type' => 'string'],
                    'subtitle' => ['type' => 'string'],
                    'placeholder' => ['type' => 'string'],
                    'class' => ['type' => 'string'],
                ],
            ]
        );

        register_block_type(
            'crashcount/council-map-tout',
            [
                'api_version' => 2,
                'editor_script' => self::BLOCK_EDITOR_HANDLE,
                'render_callback' => function (array $attributes = []): string {
                    return $this->render_council_map_shortcode($attributes);
                },
                'attributes' => [
                    'base_url' => ['type' => 'string'],
                    'mode' => ['type' => 'string'],
                    'width' => ['type' => 'string'],
                    'height' => ['type' => 'string'],
                    'aspect_ratio' => ['type' => 'string'],
                    'window' => ['type' => 'string'],
                    'headline' => ['type' => 'string'],
                    'as_of_text' => ['type' => 'string'],
                    'class' => ['type' => 'string'],
                ],
            ]
        );

        register_block_type(
            'crashcount/crashcounter-strip',
            [
                'api_version' => 2,
                'editor_script' => self::BLOCK_EDITOR_HANDLE,
                'render_callback' => function (array $attributes = []): string {
                    return $this->render_strip_shortcode($attributes);
                },
                'attributes' => [
                    'base_url' => ['type' => 'string'],
                    'layer' => ['type' => 'string'],
                    'id' => ['type' => 'string'],
                    'window' => ['type' => 'string'],
                    'show_as_of' => ['type' => 'string'],
                    'class' => ['type' => 'string'],
                ],
            ]
        );

        register_block_type(
            'crashcount/find-results',
            [
                'api_version' => 2,
                'editor_script' => self::BLOCK_EDITOR_HANDLE,
                'render_callback' => function (array $attributes = []): string {
                    return $this->render_find_results_shortcode($attributes);
                },
                'attributes' => [
                    'base_url' => ['type' => 'string'],
                    'url' => ['type' => 'string'],
                    'sid' => ['type' => 'string'],
                    'height' => ['type' => 'string'],
                    'class' => ['type' => 'string'],
                ],
            ]
        );
    }

    public function enqueue_block_editor_assets(): void {
        $this->register_block_editor_script();
        wp_enqueue_script(self::BLOCK_EDITOR_HANDLE);
    }

    private function register_block_editor_script(): void {
        if (!wp_script_is(self::BLOCK_EDITOR_HANDLE, 'registered')) {
            wp_register_script(
                self::BLOCK_EDITOR_HANDLE,
                '',
                ['wp-blocks', 'wp-element', 'wp-components', 'wp-block-editor', 'wp-i18n'],
                self::VERSION,
                true
            );
        }
        if (!self::$block_editor_script_inlined) {
            wp_add_inline_script(self::BLOCK_EDITOR_HANDLE, $this->block_editor_inline_script());
            self::$block_editor_script_inlined = true;
        }
    }

    private function block_editor_inline_script(): string {
        return <<<'JS'
(function (wp) {
  if (!wp || !wp.blocks || !wp.element || !wp.components) return;
  const registerBlockType = wp.blocks.registerBlockType;
  const createBlock = wp.blocks.createBlock;
  const el = wp.element.createElement;
  const Fragment = wp.element.Fragment;
  const InspectorControls = (wp.blockEditor && wp.blockEditor.InspectorControls) || (wp.editor && wp.editor.InspectorControls);
  const PanelBody = wp.components.PanelBody;
  const TextControl = wp.components.TextControl;
  const SelectControl = wp.components.SelectControl;
  const Notice = wp.components.Notice;

  function controls(attrs, setAttributes, fields) {
    if (!InspectorControls) return null;
    return el(
      InspectorControls,
      {},
      el(
        PanelBody,
        { title: 'Embed settings', initialOpen: true },
        fields.map((field) => {
          if (field.type === 'select') {
            return el(SelectControl, {
              key: field.key,
              label: field.label,
              value: attrs[field.key] || field.defaultValue || '',
              options: field.options || [],
              onChange: (value) => setAttributes({ [field.key]: value })
            });
          }
          return el(TextControl, {
            key: field.key,
            label: field.label,
            value: attrs[field.key] || '',
            onChange: (value) => setAttributes({ [field.key]: value })
          });
        })
      )
    );
  }

  function transformFromShortcode(tag, blockName) {
    return {
      from: [
        {
          type: 'shortcode',
          tag: tag,
          transform: function (attrs) {
            const named = attrs && attrs.named ? attrs.named : {};
            return createBlock(blockName, named);
          }
        }
      ]
    };
  }

  function makeEdit(description, fields) {
    return function (props) {
      return el(
        Fragment,
        {},
        controls(props.attributes, props.setAttributes, fields),
        el(
          'div',
          { style: { border: '1px solid #d0d7e2', borderRadius: '10px', padding: '12px', background: '#fff' } },
          el(Notice, { status: 'info', isDismissible: false }, description)
        )
      );
    };
  }

  registerBlockType('crashcount/crashfinder', {
    title: 'CrashCount: Crash Finder',
    icon: 'search',
    category: 'widgets',
    attributes: {
      title: { type: 'string', default: 'Try Crash Finder' },
      subtitle: { type: 'string', default: 'Look up any street, school, address, or intersection to see how safe the streets are.' },
      placeholder: { type: 'string', default: 'Search NYC location' },
      class: { type: 'string', default: '' }
    },
    transforms: transformFromShortcode('crashcount_crashfinder', 'crashcount/crashfinder'),
    edit: makeEdit('Crash Finder widget renders on the published page.', [
      { key: 'title', label: 'Title' },
      { key: 'subtitle', label: 'Subtitle' },
      { key: 'placeholder', label: 'Placeholder' },
      { key: 'class', label: 'CSS class' }
    ]),
    save: function () { return null; }
  });

  registerBlockType('crashcount/council-map-tout', {
    title: 'CrashCount: Council Map Tout',
    icon: 'location-alt',
    category: 'widgets',
    attributes: {
      mode: { type: 'string', default: 'image' },
      width: { type: 'string', default: '400px' },
      height: { type: 'string', default: '400px' },
      aspect_ratio: { type: 'string', default: '1 / 1' },
      window: { type: 'string', default: '' },
      headline: { type: 'string', default: '' },
      as_of_text: { type: 'string', default: '' },
      class: { type: 'string', default: '' }
    },
    transforms: transformFromShortcode('crashcount_council_map_tout', 'crashcount/council-map-tout'),
    edit: makeEdit('Council map tout renders on publish. Use image mode by default.', [
      { key: 'mode', label: 'Mode', type: 'select', defaultValue: 'image', options: [
        { label: 'Image (recommended)', value: 'image' },
        { label: 'Iframe (live map)', value: 'iframe' }
      ] },
      { key: 'width', label: 'Width' },
      { key: 'height', label: 'Height' },
      { key: 'aspect_ratio', label: 'Aspect ratio' },
      { key: 'window', label: 'Window (optional)' },
      { key: 'headline', label: 'Headline (optional)' },
      { key: 'as_of_text', label: 'As of text (optional)' },
      { key: 'class', label: 'CSS class' }
    ]),
    save: function () { return null; }
  });

  registerBlockType('crashcount/crashcounter-strip', {
    title: 'CrashCount: CrashCounter Strip',
    icon: 'minus',
    category: 'widgets',
    attributes: {
      layer: { type: 'string', default: 'citywide' },
      id: { type: 'string', default: 'nyc' },
      window: { type: 'string', default: 'auto' },
      class: { type: 'string', default: '' }
    },
    transforms: transformFromShortcode('crashcount_crashcounter_strip', 'crashcount/crashcounter-strip'),
    edit: makeEdit('Compact crash summary strip renders on publish.', [
      { key: 'layer', label: 'Layer', type: 'select', defaultValue: 'citywide', options: [
        { label: 'Citywide', value: 'citywide' },
        { label: 'Borough', value: 'borough' },
        { label: 'Council', value: 'council' },
        { label: 'Assembly', value: 'assembly' },
        { label: 'Senate', value: 'senate' },
        { label: 'Precinct', value: 'precinct' },
        { label: 'Community Board', value: 'community' },
        { label: 'Neighborhood (NTA)', value: 'nta' }
      ] },
      { key: 'id', label: 'Geography ID (example: 39, brooklyn, qn0202)' },
      { key: 'window', label: 'Window key (or auto)' },
      { key: 'class', label: 'CSS class' }
    ]),
    save: function () { return null; }
  });

  registerBlockType('crashcount/find-results', {
    title: 'CrashCount: Find Results Card',
    icon: 'media-document',
    category: 'widgets',
    attributes: {
      url: { type: 'string', default: '' },
      sid: { type: 'string', default: '' },
      height: { type: 'string', default: '230px' },
      class: { type: 'string', default: '' }
    },
    transforms: transformFromShortcode('crashcount_find_results', 'crashcount/find-results'),
    edit: makeEdit('Saved Crash Finder results card renders on publish.', [
      { key: 'url', label: 'Find URL (optional)' },
      { key: 'sid', label: 'Snapshot ID (optional)' },
      { key: 'height', label: 'Embed height' },
      { key: 'class', label: 'CSS class' }
    ]),
    save: function () { return null; }
  });
})(window.wp);
JS;
    }

    private function enqueue_assets(array $attrs): void {
        $base_url = untrailingslashit(esc_url_raw((string) ($attrs['base_url'] ?? '')));
        if ($base_url === '') {
            $base_url = 'https://crashcountnyc.com';
        }
        $asset_base = untrailingslashit(esc_url_raw((string) ($attrs['asset_base'] ?? '')));
        if ($asset_base === '') {
            $asset_base = $base_url;
        }
        $script_src = esc_url_raw((string) ($attrs['script_src'] ?? ''));
        if ($script_src === '') {
            $script_src = $asset_base . '/js/crashfinder-widget.js';
        }
        $core_src = $asset_base . '/js/crashfinder-search-core.js';
        $index_src = esc_url_raw((string) ($attrs['index_src'] ?? ''));
        if ($index_src === '') {
            $index_src = $asset_base . '/js/crashfinder-search-index.js';
        }
        $style_src = esc_url_raw((string) ($attrs['style_src'] ?? ''));
        if ($style_src === '') {
            $style_src = $asset_base . '/css/crashfinder-widget.css';
        }
        $asset_version = sanitize_text_field((string) ($attrs['asset_version'] ?? self::VERSION));
        if ($asset_version === '') {
            $asset_version = self::VERSION;
        }

        wp_register_script(self::INDEX_HANDLE, $index_src, [], $asset_version, true);
        wp_register_script(self::CORE_HANDLE, $core_src, [], $asset_version, true);
        wp_register_script(self::SCRIPT_HANDLE, $script_src, [self::INDEX_HANDLE, self::CORE_HANDLE], $asset_version, true);
        wp_register_style(self::STYLE_HANDLE, $style_src, [], $asset_version);
        wp_enqueue_script(self::INDEX_HANDLE);
        wp_enqueue_script(self::CORE_HANDLE);
        wp_enqueue_script(self::SCRIPT_HANDLE);
        wp_enqueue_style(self::STYLE_HANDLE);
    }

    private function sanitize_css_size(string $value, string $fallback): string {
        $trimmed = trim($value);
        if ($trimmed === '') {
            return $fallback;
        }
        if (strtolower($trimmed) === 'auto') {
            return 'auto';
        }
        if (preg_match('/^\d+$/', $trimmed)) {
            return $trimmed . 'px';
        }
        if (preg_match('/^\d+(?:\.\d+)?(?:px|rem|em|vw|vh|%)$/i', $trimmed)) {
            return strtolower($trimmed);
        }
        return $fallback;
    }

    private function sanitize_css_aspect_ratio(string $value, string $fallback): string {
        $trimmed = trim($value);
        if ($trimmed === '') {
            return $fallback;
        }
        if (preg_match('/^\d+(?:\.\d+)?\s*\/\s*\d+(?:\.\d+)?$/', $trimmed)) {
            return preg_replace('/\s+/', ' ', $trimmed);
        }
        return $fallback;
    }

    private function sanitize_bool(string $value, bool $fallback = true): bool {
        $normalized = strtolower(trim($value));
        if ($normalized === '') {
            return $fallback;
        }
        if (in_array($normalized, ['1', 'true', 'yes', 'on'], true)) {
            return true;
        }
        if (in_array($normalized, ['0', 'false', 'no', 'off'], true)) {
            return false;
        }
        return $fallback;
    }

    private function sanitize_strip_layer(string $value): string {
        $normalized = strtolower(trim($value));
        if (in_array($normalized, ['citywide', 'borough', 'council', 'assembly', 'senate', 'precinct', 'community', 'nta'], true)) {
            return $normalized;
        }
        return 'citywide';
    }

    private function sanitize_strip_geo_id(string $value): string {
        $normalized = strtolower(trim($value));
        if ($normalized === '') {
            return 'nyc';
        }
        $normalized = preg_replace('/[^a-z0-9_-]+/', '', $normalized);
        if (!is_string($normalized) || $normalized === '') {
            return 'nyc';
        }
        return str_replace('_', '-', $normalized);
    }

    private function sanitize_window_value(string $value): string {
        $normalized = strtolower(trim($value));
        if ($normalized === '' || $normalized === 'auto') {
            return 'auto';
        }
        $normalized = preg_replace('/[^a-z0-9_-]+/', '', $normalized);
        return is_string($normalized) ? $normalized : 'auto';
    }

    private function fallback_strip_name(string $layer, string $geo_id): string {
        if ($layer === 'citywide') {
            return 'New York City';
        }
        if ($layer === 'borough') {
            return ucwords(str_replace('-', ' ', str_replace('_', ' ', $geo_id)));
        }
        if ($layer === 'precinct') {
            return 'Precinct ' . $geo_id;
        }
        if ($layer === 'council') {
            return 'Council District ' . $geo_id;
        }
        if ($layer === 'assembly') {
            return 'Assembly District ' . $geo_id;
        }
        if ($layer === 'senate') {
            return 'Senate District ' . $geo_id;
        }
        if ($layer === 'community') {
            return 'Community Board ' . $geo_id;
        }
        if ($layer === 'nta') {
            return strtoupper($geo_id);
        }
        return $geo_id;
    }

    private function is_local_host_url(string $base_url): bool {
        $host = strtolower((string) parse_url($base_url, PHP_URL_HOST));
        if ($host === '') {
            return true;
        }
        if (in_array($host, ['localhost', '127.0.0.1', '::1'], true)) {
            return true;
        }
        if (str_ends_with($host, '.local') || str_ends_with($host, '.test')) {
            return true;
        }
        return false;
    }

    private function extract_sid_from_url(string $url): string {
        $url = trim($url);
        if ($url === '') {
            return '';
        }
        $parts = wp_parse_url($url);
        if (!is_array($parts)) {
            return '';
        }
        $query = $parts['query'] ?? '';
        if (!is_string($query) || $query === '') {
            return '';
        }
        parse_str($query, $params);
        return sanitize_text_field((string) ($params['sid'] ?? ''));
    }

    private function enqueue_council_map_styles(): void {
        if (self::$map_tout_style_enqueued) {
            return;
        }

        $css = <<<'CSS'
.crashcount-council-map-tout {
  width: var(--crashcount-council-map-width, 100%);
  max-width: 100%;
  border: 1px solid #d5d9e0;
  border-radius: 14px;
  background: #fff;
  overflow: hidden;
  margin: 0;
  box-shadow: 0 10px 28px rgba(15, 23, 42, 0.08);
  font-family: "Avenir Next LT Pro", "Avenir Next", Avenir, -apple-system, BlinkMacSystemFont, "Segoe UI", Helvetica, Arial, sans-serif;
  color: #0f172a;
}
.crashcount-council-map-tout--frame {
  border: 0;
  border-radius: 0;
  background: transparent;
  overflow: visible;
}
.crashcount-council-map-tout__frame {
  display: block;
  width: var(--crashcount-council-map-width, 100%);
  max-width: 100%;
  height: var(--crashcount-council-map-height, 420px);
  border: 0;
  border-radius: 14px;
}
.crashcount-council-map-tout__header {
  border-bottom: 1px solid #d5d9e0;
  padding: 0.75rem 1rem 0.82rem;
  background: #ffffff;
}
.crashcount-council-map-tout .crashcount-council-map-tout__header h3 {
  margin: 0;
  font-size: clamp(1.55rem, 3.4vw, 2.1rem) !important;
  font-weight: 700 !important;
  letter-spacing: -0.018em;
  line-height: 1.06 !important;
  text-align: center;
  text-wrap: balance;
  color: #0f172a !important;
  font-family: inherit !important;
}
.crashcount-council-map-tout__link {
  position: relative;
  display: block;
  color: inherit;
  text-decoration: none;
}
.crashcount-council-map-tout__image {
  display: block;
  width: 100%;
  height: var(--crashcount-council-map-height, auto);
  min-height: 220px;
  aspect-ratio: var(--crashcount-council-map-aspect-ratio, 1 / 1);
  object-fit: cover;
  object-position: center center;
}
.crashcount-council-map-tout__copy {
  padding: 0.75rem 0.9rem 0.85rem;
  border-top: 1px solid #d5d9e0;
  background: #fff;
}
.crashcount-council-map-tout__headline {
  margin: 0;
  color: #0f172a;
  font-size: 0.95rem;
  line-height: 1.35;
  font-weight: 700;
}
.crashcount-council-map-tout__asof {
  margin: 0.35rem 0 0;
  color: #64748b;
  font-size: 0.8rem;
  line-height: 1.3;
}
.crashcount-council-map-tout__button {
  position: absolute;
  top: 1rem;
  right: 1rem;
  background: #07163a;
  color: #fff;
  border-radius: 999px;
  padding: 0.58rem 0.98rem;
  font-weight: 700;
  font-size: 0.93rem;
  letter-spacing: -0.01em;
  line-height: 1;
  box-shadow: 0 8px 22px rgba(7, 22, 58, 0.32);
  transition: background-color 120ms ease, transform 120ms ease;
}
.crashcount-council-map-tout__link:hover .crashcount-council-map-tout__button,
.crashcount-council-map-tout__link:focus-visible .crashcount-council-map-tout__button {
  background: #0d2564;
  transform: translateY(-1px);
}
@media (max-width: 680px) {
  .crashcount-council-map-tout .crashcount-council-map-tout__header h3 {
    font-size: 1.45rem !important;
  }
  .crashcount-council-map-tout__button {
    top: 0.7rem;
    right: 0.7rem;
    font-size: 0.86rem;
    padding: 0.48rem 0.76rem;
  }
  .crashcount-council-map-tout__headline {
    font-size: 0.88rem;
  }
  .crashcount-council-map-tout__asof {
    font-size: 0.76rem;
  }
}
CSS;

        wp_register_style(self::MAP_TOUT_STYLE_HANDLE, false, [], self::VERSION);
        wp_enqueue_style(self::MAP_TOUT_STYLE_HANDLE);
        wp_add_inline_style(self::MAP_TOUT_STYLE_HANDLE, $css);
        self::$map_tout_style_enqueued = true;
    }

    private function enqueue_strip_styles(): void {
        if (self::$strip_style_enqueued) {
            return;
        }
        $css = <<<'CSS'
.crashcount-crashcounter-strip {
  margin: 0;
}
.crashcount-strip__link {
  display: block;
  border: 1px solid #d0d7e2;
  border-radius: 10px;
  background: #fff;
  color: #0f172a;
  text-decoration: none;
  padding: 0.62rem 0.8rem 0.66rem;
  font-family: "Avenir Next LT Pro", "Avenir Next", Avenir, -apple-system, BlinkMacSystemFont, "Segoe UI", Helvetica, Arial, sans-serif;
  line-height: 1.24;
}
.crashcount-strip__link:hover,
.crashcount-strip__link:focus-visible {
  border-color: #94a3b8;
  box-shadow: 0 8px 20px rgba(15, 23, 42, 0.08);
}
.crashcount-strip__headline {
  margin: 0;
  font-size: 0.98rem;
  font-weight: 700;
  letter-spacing: -0.01em;
}
.crashcount-strip__meta {
  margin: 0.28rem 0 0;
  color: #64748b;
  font-size: 0.78rem;
}
@media (max-width: 760px) {
  .crashcount-strip__headline {
    font-size: 0.9rem;
  }
  .crashcount-strip__meta {
    font-size: 0.74rem;
  }
}
CSS;
        wp_register_style(self::STRIP_STYLE_HANDLE, false, [], self::VERSION);
        wp_enqueue_style(self::STRIP_STYLE_HANDLE);
        wp_add_inline_style(self::STRIP_STYLE_HANDLE, $css);
        self::$strip_style_enqueued = true;
    }

    private function enqueue_find_results_styles(): void {
        if (self::$find_results_style_enqueued) {
            return;
        }
        $css = <<<'CSS'
.crashcount-find-results {
  margin: 0;
}
.crashcount-find-results__frame {
  display: block;
  width: 100%;
  border: 0;
  border-radius: 14px;
}
CSS;
        wp_register_style(self::FIND_RESULTS_STYLE_HANDLE, false, [], self::VERSION);
        wp_enqueue_style(self::FIND_RESULTS_STYLE_HANDLE);
        wp_add_inline_style(self::FIND_RESULTS_STYLE_HANDLE, $css);
        self::$find_results_style_enqueued = true;
    }

    private function fetch_council_map_summary(string $base_url, string $summary_path): array {
        $path = trim($summary_path);
        if ($path === '') {
            $path = self::COUNCIL_MAP_SUMMARY_PATH;
        }
        if ($path[0] !== '/') {
            $path = '/' . $path;
        }

        $cache_key = 'crashcount_council_summary_' . md5($base_url . $path);
        $cached = get_transient($cache_key);
        if (is_array($cached)) {
            return $cached;
        }

        $url = $base_url . $path;
        $response = wp_remote_get($url, [
            'timeout' => 4,
            'redirection' => 2,
        ]);
        if (is_wp_error($response)) {
            return [];
        }

        $status = (int) wp_remote_retrieve_response_code($response);
        if ($status < 200 || $status >= 300) {
            return [];
        }

        $body = wp_remote_retrieve_body($response);
        if (!is_string($body) || $body === '') {
            return [];
        }

        $decoded = json_decode($body, true);
        if (!is_array($decoded)) {
            return [];
        }

        $summary = [
            'window_key' => sanitize_text_field((string) ($decoded['window_key'] ?? '')),
            'headline' => sanitize_text_field((string) ($decoded['headline'] ?? '')),
            'as_of_text' => sanitize_text_field((string) ($decoded['as_of_text'] ?? '')),
        ];
        set_transient($cache_key, $summary, self::COUNCIL_MAP_SUMMARY_CACHE_SECONDS);
        return $summary;
    }

    private function fetch_strip_summary(string $base_url, string $summary_path, string $layer, string $geo_id): array {
        $path = trim($summary_path);
        if ($path === '') {
            $path = self::STRIP_SUMMARY_PATH;
        }
        if ($path[0] !== '/') {
            $path = '/' . $path;
        }
        $layer = $this->sanitize_strip_layer($layer);
        $geo_id = $this->sanitize_strip_geo_id($geo_id);
        $url = sprintf('%s%s/%s/%s.json', $base_url, rtrim($path, '/'), $layer, $geo_id);

        $cache_key = 'crashcount_strip_summary_' . md5($url);
        $cached = get_transient($cache_key);
        if (is_array($cached)) {
            return $cached;
        }

        $response = wp_remote_get($url, [
            'timeout' => 4,
            'redirection' => 2,
        ]);
        if (is_wp_error($response)) {
            return [];
        }
        $status = (int) wp_remote_retrieve_response_code($response);
        if ($status < 200 || $status >= 300) {
            return [];
        }
        $body = wp_remote_retrieve_body($response);
        if (!is_string($body) || $body === '') {
            return [];
        }
        $decoded = json_decode($body, true);
        if (!is_array($decoded)) {
            return [];
        }

        $windows = [];
        if (isset($decoded['windows']) && is_array($decoded['windows'])) {
            foreach ($decoded['windows'] as $window_key => $window_payload) {
                if (!is_array($window_payload)) {
                    continue;
                }
                $metrics_raw = isset($window_payload['metrics']) && is_array($window_payload['metrics'])
                    ? $window_payload['metrics']
                    : [];
                $windows[sanitize_text_field((string) $window_key)] = [
                    'window_key' => sanitize_text_field((string) ($window_payload['window_key'] ?? $window_key)),
                    'window_label' => sanitize_text_field((string) ($window_payload['window_label'] ?? $window_key)),
                    'metrics' => [
                        'crashes' => (int) ($metrics_raw['crashes'] ?? 0),
                        'injuries' => (int) ($metrics_raw['injuries'] ?? 0),
                        'serious_injuries' => (int) ($metrics_raw['serious_injuries'] ?? 0),
                        'deaths' => (int) ($metrics_raw['deaths'] ?? 0),
                    ],
                ];
            }
        }
        $metrics = isset($decoded['metrics']) && is_array($decoded['metrics']) ? $decoded['metrics'] : [];
        $summary = [
            'name' => sanitize_text_field((string) ($decoded['name'] ?? '')),
            'window_key' => sanitize_text_field((string) ($decoded['window_key'] ?? '')),
            'window_label' => sanitize_text_field((string) ($decoded['window_label'] ?? '')),
            'as_of_text' => sanitize_text_field((string) ($decoded['as_of_text'] ?? '')),
            'target_url' => sanitize_text_field((string) ($decoded['target_url'] ?? '/')),
            'metrics' => [
                'crashes' => (int) ($metrics['crashes'] ?? 0),
                'injuries' => (int) ($metrics['injuries'] ?? 0),
                'serious_injuries' => (int) ($metrics['serious_injuries'] ?? 0),
                'deaths' => (int) ($metrics['deaths'] ?? 0),
            ],
            'windows' => $windows,
        ];

        set_transient($cache_key, $summary, self::STRIP_SUMMARY_CACHE_SECONDS);
        return $summary;
    }

    private function resolve_tracking_params(): array {
        $tracking = self::DEFAULT_TRACKING;
        $filtered = apply_filters('crashcount_crashfinder_tracking', $tracking);
        if (!is_array($filtered)) {
            $filtered = $tracking;
        }

        $normalized = [];
        foreach ($tracking as $key => $default_value) {
            $raw_value = $filtered[$key] ?? $default_value;
            $normalized[$key] = sanitize_text_field((string) $raw_value);
        }

        return $normalized;
    }

    public function register_privacy_policy_content(): void {
        if (!function_exists('wp_add_privacy_policy_content')) {
            return;
        }

        $content = sprintf(
            '<p>%1$s</p><ul><li>%2$s</li><li>%3$s</li><li>%4$s</li></ul><p>%5$s</p>',
            esc_html__('CrashCount Crash Finder Shortcode loads embeds from crashcountnyc.com and can request live crash summary data when shortcode or block output is rendered on a page.', 'crashcount-crashfinder-shortcode'),
            esc_html__('When pages with these embeds are viewed, visitor browsers request scripts/styles/images/JSON from crashcountnyc.com. Standard web request metadata (such as IP address, user agent, and referrer) is sent by the browser.', 'crashcount-crashfinder-shortcode'),
            esc_html__('When visitors use Crash Finder search, the typed search query is sent to CrashCount endpoints to return location suggestions and the matching Crash Finder page.', 'crashcount-crashfinder-shortcode'),
            esc_html__('Outbound links may include source and UTM parameters for attribution. Site owners can override or clear these via the crashcount_crashfinder_tracking filter.', 'crashcount-crashfinder-shortcode'),
            sprintf(
                /* translators: 1: Privacy Policy URL, 2: Terms URL. */
                esc_html__('CrashCount privacy policy: %1$s. CrashCount terms: %2$s.', 'crashcount-crashfinder-shortcode'),
                esc_url(self::PRIVACY_POLICY_URL),
                esc_url(self::TERMS_URL)
            )
        );

        wp_add_privacy_policy_content(
            esc_html__('CrashCount Crash Finder Shortcode', 'crashcount-crashfinder-shortcode'),
            wp_kses_post(wpautop($content))
        );
    }
}

new CrashCount_CrashFinder_Shortcode();
